<?php

namespace App\Http\Controllers\Backend;

use App\DataTables\ProductDataTable;
use App\Http\Controllers\Controller;
use App\Http\Requests\Backend\ProductStoreRequest;
use App\Http\Requests\Backend\ProductUpdateRequest;
use App\Models\Category;
use Illuminate\Support\Facades\File;
use App\Models\Product;
use Illuminate\Http\Request;
use Intervention\Image\Facades\Image;
use Yajra\DataTables\DataTables;

class ProductManagementController extends Controller
{

    public function index(ProductDataTable $dataTable)
    {
        return $dataTable->render('backend.pages.products.index');

    }

    public function create()
    {
        $categories = Category::with('categories')->get();

        $products = Product::all();

        return view('backend.pages.products.create', compact('categories', 'products'));
    }

    public function store(ProductStoreRequest $request)
    {
       // dd($request->all());
        $product = new Product();

        $product->product_title       = $request->product_title;
        $product->product_name        = $request->product_name;
        $product->slug                = $request->slug;
        $product->product_description = $request->product_description ?? '';
        $product->product_detail      = $request->product_detail ?? '';
        $product->meta_keywords       = $request->meta_keywords;
        $product->meta_description    = $request->meta_description;
        $product->meta_title          = $request->meta_title;
        $product->by_style            = $request->by_style ? 1 : 0;
        $product->by_material         = $request->by_material ? 1 : 0;
        $product->by_industry         = $request->by_industry ? 1 : 0;
        $product->display_homepage    = $request->display_homepage ? 1 : 0;
        $product->status              = $request->status ?? 1;
        $product->faqs =              json_encode($request->faqs ?? []);
        // Note: client_story will be processed later in the method

        // Handle related products
        $related_products = $request->related_products ?? [];
        $product->related_products = json_encode($related_products);

        $productFrontImage = $request->file('product_front_image');
        $productBackImage =  $request->file('product_back_image');
        $productMultipleImage =  $request->file('product_multiple_images');

        if ($productFrontImage){
            $product->addMedia($productFrontImage)->toMediaCollection('product_front_image');
        }
        if ($productBackImage){
            $product->addMedia($productBackImage)->toMediaCollection('product_back_image');
        }

        if ($productMultipleImage)
        {
            foreach ($request->file('product_multiple_images') as $image)
            {
                $product->addMedia($image)->toMediaCollection('product_gallery');
            }
        }

        // Handle client story images
        if ($request->has('client_story')) {
            $clientStories = $request->client_story;
            $processedStories = [];

            foreach ($clientStories as $key => $story) {
                $processedStory = [
                    'description' => $story['description'] ?? ''
                ];

                // Handle image upload if exists
                if (isset($story['image']) && $story['image']) {
                    $imagePath = $story['image']->store('client_stories', 'public');
                    $processedStory['image'] = $imagePath;
                } else {
                    // Set empty image if no image provided
                    $processedStory['image'] = '';
                }

                $processedStories[$key] = $processedStory;
            }

            $product->client_story = json_encode($processedStories);
        } else {
            // If no client_story data provided, set empty array
            $product->client_story = json_encode([]);
        }

          $product->save();
        \Cache::forget('homepage_products_with_category');

        $product->category()->attach($request->product_category_id);

        return redirect()->route('dashboard.product.index')->with('success','Product added Successfully');

    }

    public function edit(Product $product)
    {

        $categories = Category::with('categories')->get();

        $products = Product::all();

        $images =  json_decode($product->product_multiple_images);

        $related_products= json_decode($product->related_products);


        return view('backend.pages.products.edit',compact('product','categories','images','related_products','products'));
    }

    public function update(ProductUpdateRequest $request, Product $product)
    {
        // Update product attributes
        $product->product_title       = $request->product_title;
        $product->product_name        = $request->product_name;
        $product->slug        = $request->slug;
        $product->product_description = $request->product_description ?? '';
        $product->product_detail      = $request->product_detail ?? '';
        $product->meta_keywords       = $request->meta_keywords;
        $product->meta_description    = $request->meta_description;
        $product->meta_title          = $request->meta_title;
        $product->by_style            = $request->by_style ? 1 : 0;
        $product->by_material         = $request->by_material ? 1 : 0;
        $product->by_industry         = $request->by_industry ? 1 : 0;
        $product->status              = $request->status ?? 1;
        $product->display_homepage    = $request->display_homepage ? 1 : 0;

        // Handle related products
        $related_products = $request->related_products ?? [];
        $product->related_products = json_encode($related_products);
        $product->faqs =              json_encode($request->faqs ?? []);
        // Note: client_story will be processed later in the method

        // Handle media uploads
        $productFrontImage = $request->file('product_front_image');
        $productBackImage = $request->file('product_back_image');
        $productMultipleImages = $request->file('product_multiple_images');

        // Update front image
        if ($productFrontImage) {
            // Clear existing front image
            $product->clearMediaCollection('product_front_image');
            // Add new front image
            $product->addMedia($productFrontImage)->toMediaCollection('product_front_image');
        }

        // Update back image
        if ($productBackImage) {
            // Clear existing back image
            $product->clearMediaCollection('product_back_image');
            // Add new back image
            $product->addMedia($productBackImage)->toMediaCollection('product_back_image');
        }

        // Update multiple images
        if ($productMultipleImages) {
            // Clear existing gallery images
            $product->clearMediaCollection('product_gallery');
            // Add new multiple images
            foreach ($productMultipleImages as $image) {
                $product->addMedia($image)->toMediaCollection('product_gallery');
            }
        }

        // Handle client story images
        if ($request->has('client_story')) {
            $clientStories = $request->client_story;
            $processedStories = [];

            // Get existing client stories to preserve existing images if no new image uploaded
            $existingStories = json_decode($product->client_story, true) ?? [];

            foreach ($clientStories as $key => $story) {
                $processedStory = [
                    'description' => $story['description'] ?? ''
                ];

                // Handle image upload if exists
                if (isset($story['image']) && $story['image']) {
                    $imagePath = $story['image']->store('client_stories', 'public');
                    $processedStory['image'] = $imagePath;
                } else {
                    // Keep existing image if no new image uploaded
                    $processedStory['image'] = $existingStories[$key]['image'] ?? '';
                }

                $processedStories[$key] = $processedStory;
            }

            $product->client_story = json_encode($processedStories);
        } else {
            // If no client_story data provided, keep existing data
            // This handles cases where the form doesn't include client_story fields
            if (!isset($product->client_story)) {
                $product->client_story = json_encode([]);
            }
        }

        // Save the product
        $product->save();

        // Sync categories
        $category = $request->input('product_category_id') ? $request->input('product_category_id') : [];
        $product->category()->sync($category);
        \Cache::forget('homepage_products_with_category');
        return redirect()->route('dashboard.product.index')->with('success', 'Product Updated Successfully');
    }

    public function delete(Product $product)
    {
        $product->delete();
        $product->category()->sync([]);
        $product->clearMediaCollection('product_back_image');
        $product->clearMediaCollection('product_front_image');
        $product->clearMediaCollection('product_gallery');
        return back()->with('success', 'Product Deleted Successfully');
    }

    public function ProductDetail(Request $request)
    {

        $product_id = $request->id;
        $productDetail = Product::where('id', $product_id)->with('category')->first();
        $images = json_decode($productDetail->product_multiple_images);
        return view('backend.pages.products.productDetailModel', compact('productDetail', 'images'));

    }

    public function deleteMedia($slug, $mediaId)
    {
        $product = Product::where('slug', $slug)->firstOrFail();
        $media = $product->getMedia('product_gallery')->find($mediaId);
        if ($media) {
            $media->delete();
        }
        return back()->with('success', 'Media deleted successfully.');
    }

}
