<?php

namespace App\Services;

use Artesaos\SEOTools\Facades\SEOTools;
use Artesaos\SEOTools\Facades\TwitterCard;

class SeoService
{
    /**
     * Set SEO metadata for the page
     *
     * @param string $title Page title
     * @param string $description Meta description
     * @param string $keywords Meta keywords
     * @param string $url Canonical URL
     * @param string|null $image OG image URL
     * @param bool $indexable Whether the page should be indexed by search engines
     * @param string|null $twitterHandle Twitter handle for Twitter Card
     * @return void
     */
    public function setMetaData(
        string $title,
        ?string $description,
        ?string $keywords,
        string $url,
        ?string $image = null,
        bool $indexable = false,
        ?string $twitterHandle = null
    ): void {
        // Set defaults for optional parameters
        $description = $description ?? '';
        $keywords = $keywords ?? '';
        $image = $image ?? asset('og_image.jpeg');
        $twitterHandle = $twitterHandle ?? '@customboxeszone';

        // Basic SEO
        SEOTools::setTitle($title);
        SEOTools::setDescription($description);
        SEOTools::setCanonical($url);

        // OpenGraph
        SEOTools::opengraph()->setTitle($title);
        SEOTools::opengraph()->setDescription($description);
        SEOTools::opengraph()->setSiteName(config('app.name'));
        SEOTools::opengraph()->setUrl($url);
        SEOTools::opengraph()->addProperty('type', 'website');
        SEOTools::opengraph()->addProperty('locale', 'en_US');
        SEOTools::opengraph()->addProperty('image:alt', config('app.name'));
        SEOTools::opengraph()->addImage($image);

        // Twitter Card
        TwitterCard::setType('summary_large_image');
        TwitterCard::setTitle($title);
        TwitterCard::setDescription($description);
        TwitterCard::setUrl($url);
        TwitterCard::setImage($image);
        TwitterCard::setSite($twitterHandle);

        // Additional Meta Tags
        SEOTools::metatags()->addMeta('author', config('app.name'));
        SEOTools::metatags()->addMeta('robots', $indexable ? 'index, follow' : 'noindex, nofollow');
        SEOTools::metatags()->setKeywords([$keywords]);
    }

    /**
     * Set SEO metadata with custom meta tags
     *
     * @param string $title Page title
     * @param string $description Meta description
     * @param string $keywords Meta keywords
     * @param string $url Canonical URL
     * @param string|null $image OG image URL
     * @param array $options Additional options (indexable, twitterHandle, customMeta)
     * @return void
     */
    public function setMetaDataWithOptions(
        string $title,
        ?string $description,
        ?string $keywords,
        string $url,
        ?string $image = null,
        array $options = []
    ): void {
        $indexable = $options['indexable'] ?? true;
        $twitterHandle = $options['twitterHandle'] ?? null;

        // Call the main method
        $this->setMetaData($title, $description, $keywords, $url, $image, $indexable, $twitterHandle);

        // Add custom meta tags if provided
        if (isset($options['customMeta']) && is_array($options['customMeta'])) {
            foreach ($options['customMeta'] as $name => $content) {
                SEOTools::metatags()->addMeta($name, $content);
            }
        }
    }
}





